# (c) Copyright 2009-2015. CodeWeavers, Inc.

import io
import sys

import distversion

from cxurlget_base import StopDownload, USER_AGENT, UrlGetterBase, url_to_basename, url_to_sanitized_basename, format_http_date, UrlError, HttpError

UrlGetter = None

if distversion.IS_MACOSX:
    from cxurlget_nsurl import UrlGetter
else:
    from cxurlget_urllib import UrlGetter

class _FetchContentHandler:
    exception = None

    def __init__(self):
        self.exception = None
        self.bytesio = io.BytesIO()
        self.writer = io.BufferedWriter(self.bytesio)

    def write(self, data):
        # We can't pass self.writer to UrlGetter directly because
        # it closes the output file, which would delete the contents.
        self.writer.write(data)

    def notify_failed(self, _urlgetter, exception):
        self.exception = exception

    def close(self):
        self.writer.flush()

    def flush(self):
        self.writer.flush()

def fetch_content(url, **kwargs):
    """Fetches a url with UrlGetter and returns it as a bytes object.

Any keyword arguments passed in will be forwarded to the UrlGetter
constructor, except for notify_failed and outfile.

Returns a bytes object containing the content.

Raises an exception on failure."""
    handler = _FetchContentHandler()
    kwargs['notify_failed'] = handler.notify_failed
    getter = UrlGetter(url, handler, **kwargs)
    getter.fetch()
    if handler.exception:
        raise handler.exception # pylint: disable=E0702
    return handler.bytesio.getvalue()

__all__ = ['StopDownload', 'USER_AGENT', 'UrlGetterBase', 'url_to_basename', 'url_to_sanitized_basename', 'format_http_date', 'UrlGetter', 'UrlError', 'HttpError', 'fetch_content']

def main():
    def notify_progress(urlgetter):
        sys.stdout.write('\r%d/%d bytes (%.1f%%)' % (urlgetter.bytes_downloaded, urlgetter.bytes_total, 100.0*urlgetter.bytes_downloaded/urlgetter.bytes_total))
        sys.stdout.flush()

    def notify_finished(_urlgetter):
        print('\nSuccess.')

    def notify_failed(_urlgetter, exception):
        print('\nFailure.')
        print('Exception: %s' % repr(exception))
        print('Message: %s' % str(exception))

    if len(sys.argv) >= 4:
        data = sys.argv[3]
        if not isinstance(data, bytes):
            data = data.encode('utf8')
    else:
        data = None

    getter = UrlGetter(sys.argv[1], sys.argv[2], notify_progress, notify_finished, notify_failed, data=data)
    getter.fetch()
    print("Suggested basename: %s" % getter.basename)

if __name__ == '__main__':
    main()
