# (c) Copyright 2009-2013. CodeWeavers, Inc.

from gi.repository import Gtk

import bottlequery

import cxguitools

# for localization
from cxutils import cxgettext as _


class MenuEditDialog:
    def __init__(self, bottlename):
        self.bottlename = bottlename

        # Import widget modules so Gtk.Builder() finds them
        import cxmenuwidget # pylint: disable=W0612,W0611
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("cxmenuedit"))
        self.xml.connect_signals(self)

        self.dialog = self.xml.get_object('menu-edit-dialog')
        self.edit_widget = self.xml.get_object('menu-editor')
        self.menuroot_entry = self.xml.get_object('menuroot_entry')

        properties = bottlequery.get_bottle_properties(bottlename)
        self.managed = properties['managed']
        if self.managed:
            self.xml.get_object("recreate_button").hide()
            self.xml.get_object("apply_button").hide()
            self.xml.get_object("cancel_button").hide()
            self.xml.get_object("ok_button").hide()
        else:
            self.xml.get_object("close_button").hide()

        self.edit_widget.set_bottle(bottlename, properties['managed'])

        self.edit_widget.connect_selection_changed(self.on_selection_changed)

        self.dialog.set_title(_('Menus in %(bottlename)s') % {'bottlename':bottlename})

        self.edit_widget.refresh(self.refresh_finished, self.refresh_fail)

        self.edit_widget.show()
        self.dialog.show()

    def on_cancel_button_clicked(self, _button):
        self.dialog.destroy()

    def on_apply_button_clicked(self, _button):
        self.disable_buttons()
        self.edit_widget.commit(self.enable_buttons, self.apply_fail)

    def on_ok_button_clicked(self, _button):
        self.disable_buttons()
        self.edit_widget.commit(self.ok_finished, self.apply_fail)

    def on_recreate_button_clicked(self, _button):
        self.disable_buttons()
        self.edit_widget.recreate_menus(self.enable_buttons, self.apply_fail)

    def on_launch_button_clicked(self, _button):
        path = self.edit_widget.get_selection()
        if path is not None:
            self.edit_widget.prefs[path].start()

    def on_selection_changed(self):
        self.xml.get_object('launch_button').set_sensitive(
            self.edit_widget.get_selection() is not None)

    def menuroot_changed(self, _widget):
        self.edit_widget.prefs.new_menuroot = self.menuroot_entry.get_text()

    def ok_finished(self):
        self.dialog.destroy()

    def apply_fail(self, error):
        self.enable_buttons()
        cxguitools.CXMessageDlg(
            primary=_("Could not save changes to menus"),
            secondary=error,
            parent=self.dialog,
            message_type=Gtk.MessageType.ERROR)

    def on_delete(self, _widget, _event):
        self.dialog.destroy()

    def disable_buttons(self):
        self.xml.get_object('recreate_button').set_sensitive(False)
        self.xml.get_object('cancel_button').set_sensitive(False)
        self.xml.get_object('apply_button').set_sensitive(False)
        self.xml.get_object('ok_button').set_sensitive(False)

    def enable_buttons(self):
        self.xml.get_object('recreate_button').set_sensitive(True)
        self.xml.get_object('cancel_button').set_sensitive(True)
        self.xml.get_object('apply_button').set_sensitive(True)
        self.xml.get_object('ok_button').set_sensitive(True)

    def refresh_finished(self):
        self.enable_buttons()
        self.menuroot_entry.set_text(self.edit_widget.prefs.menuroot)
        self.menuroot_entry.set_sensitive(not self.managed)

    def quit_message_box(self, _response):
        self.dialog.destroy()

    def refresh_fail(self, error):
        self.dialog.hide()
        cxguitools.CXMessageDlg(
            primary=_("Could not read menus for %(bottlename)s") % {'bottlename': self.bottlename},
            secondary=error,
            parent=self.dialog, response_function=self.quit_message_box,
            message_type=Gtk.MessageType.ERROR)

    def quit_requested(self, _caller):
        # pylint: disable=R0201
        cxguitools.toplevel_quit()

def start(bottle):
    MenuEditDialog(bottle.name)
