﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/ParameterInlinePolicy.h>
#include <aws/ssm/model/ParameterTier.h>
#include <aws/ssm/model/ParameterType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SSM {
namespace Model {

/**
 * <p>Information about parameter usage.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ssm-2014-11-06/ParameterHistory">AWS
 * API Reference</a></p>
 */
class ParameterHistory {
 public:
  AWS_SSM_API ParameterHistory() = default;
  AWS_SSM_API ParameterHistory(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API ParameterHistory& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the parameter.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ParameterHistory& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of parameter used.</p>
   */
  inline ParameterType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(ParameterType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline ParameterHistory& WithType(ParameterType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The alias of the Key Management Service (KMS) key used to encrypt the
   * parameter. Applies to <code>SecureString</code> parameters only</p>
   */
  inline const Aws::String& GetKeyId() const { return m_keyId; }
  inline bool KeyIdHasBeenSet() const { return m_keyIdHasBeenSet; }
  template <typename KeyIdT = Aws::String>
  void SetKeyId(KeyIdT&& value) {
    m_keyIdHasBeenSet = true;
    m_keyId = std::forward<KeyIdT>(value);
  }
  template <typename KeyIdT = Aws::String>
  ParameterHistory& WithKeyId(KeyIdT&& value) {
    SetKeyId(std::forward<KeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Date the parameter was last changed or updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedDate() const { return m_lastModifiedDate; }
  inline bool LastModifiedDateHasBeenSet() const { return m_lastModifiedDateHasBeenSet; }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  void SetLastModifiedDate(LastModifiedDateT&& value) {
    m_lastModifiedDateHasBeenSet = true;
    m_lastModifiedDate = std::forward<LastModifiedDateT>(value);
  }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  ParameterHistory& WithLastModifiedDate(LastModifiedDateT&& value) {
    SetLastModifiedDate(std::forward<LastModifiedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the Amazon Web Services user who last changed
   * the parameter.</p>
   */
  inline const Aws::String& GetLastModifiedUser() const { return m_lastModifiedUser; }
  inline bool LastModifiedUserHasBeenSet() const { return m_lastModifiedUserHasBeenSet; }
  template <typename LastModifiedUserT = Aws::String>
  void SetLastModifiedUser(LastModifiedUserT&& value) {
    m_lastModifiedUserHasBeenSet = true;
    m_lastModifiedUser = std::forward<LastModifiedUserT>(value);
  }
  template <typename LastModifiedUserT = Aws::String>
  ParameterHistory& WithLastModifiedUser(LastModifiedUserT&& value) {
    SetLastModifiedUser(std::forward<LastModifiedUserT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the parameter.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  ParameterHistory& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parameter value.</p>
   */
  inline const Aws::String& GetValue() const { return m_value; }
  inline bool ValueHasBeenSet() const { return m_valueHasBeenSet; }
  template <typename ValueT = Aws::String>
  void SetValue(ValueT&& value) {
    m_valueHasBeenSet = true;
    m_value = std::forward<ValueT>(value);
  }
  template <typename ValueT = Aws::String>
  ParameterHistory& WithValue(ValueT&& value) {
    SetValue(std::forward<ValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Parameter names can include the following letters and symbols.</p>
   * <p>a-zA-Z0-9_.-</p>
   */
  inline const Aws::String& GetAllowedPattern() const { return m_allowedPattern; }
  inline bool AllowedPatternHasBeenSet() const { return m_allowedPatternHasBeenSet; }
  template <typename AllowedPatternT = Aws::String>
  void SetAllowedPattern(AllowedPatternT&& value) {
    m_allowedPatternHasBeenSet = true;
    m_allowedPattern = std::forward<AllowedPatternT>(value);
  }
  template <typename AllowedPatternT = Aws::String>
  ParameterHistory& WithAllowedPattern(AllowedPatternT&& value) {
    SetAllowedPattern(std::forward<AllowedPatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parameter version.</p>
   */
  inline long long GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  inline void SetVersion(long long value) {
    m_versionHasBeenSet = true;
    m_version = value;
  }
  inline ParameterHistory& WithVersion(long long value) {
    SetVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Labels assigned to the parameter version.</p>
   */
  inline const Aws::Vector<Aws::String>& GetLabels() const { return m_labels; }
  inline bool LabelsHasBeenSet() const { return m_labelsHasBeenSet; }
  template <typename LabelsT = Aws::Vector<Aws::String>>
  void SetLabels(LabelsT&& value) {
    m_labelsHasBeenSet = true;
    m_labels = std::forward<LabelsT>(value);
  }
  template <typename LabelsT = Aws::Vector<Aws::String>>
  ParameterHistory& WithLabels(LabelsT&& value) {
    SetLabels(std::forward<LabelsT>(value));
    return *this;
  }
  template <typename LabelsT = Aws::String>
  ParameterHistory& AddLabels(LabelsT&& value) {
    m_labelsHasBeenSet = true;
    m_labels.emplace_back(std::forward<LabelsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parameter tier.</p>
   */
  inline ParameterTier GetTier() const { return m_tier; }
  inline bool TierHasBeenSet() const { return m_tierHasBeenSet; }
  inline void SetTier(ParameterTier value) {
    m_tierHasBeenSet = true;
    m_tier = value;
  }
  inline ParameterHistory& WithTier(ParameterTier value) {
    SetTier(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the policies assigned to a parameter.</p> <p> <a
   * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-policies.html">Assigning
   * parameter policies</a> in the <i>Amazon Web Services Systems Manager User
   * Guide</i>.</p>
   */
  inline const Aws::Vector<ParameterInlinePolicy>& GetPolicies() const { return m_policies; }
  inline bool PoliciesHasBeenSet() const { return m_policiesHasBeenSet; }
  template <typename PoliciesT = Aws::Vector<ParameterInlinePolicy>>
  void SetPolicies(PoliciesT&& value) {
    m_policiesHasBeenSet = true;
    m_policies = std::forward<PoliciesT>(value);
  }
  template <typename PoliciesT = Aws::Vector<ParameterInlinePolicy>>
  ParameterHistory& WithPolicies(PoliciesT&& value) {
    SetPolicies(std::forward<PoliciesT>(value));
    return *this;
  }
  template <typename PoliciesT = ParameterInlinePolicy>
  ParameterHistory& AddPolicies(PoliciesT&& value) {
    m_policiesHasBeenSet = true;
    m_policies.emplace_back(std::forward<PoliciesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data type of the parameter, such as <code>text</code> or
   * <code>aws:ec2:image</code>. The default is <code>text</code>.</p>
   */
  inline const Aws::String& GetDataType() const { return m_dataType; }
  inline bool DataTypeHasBeenSet() const { return m_dataTypeHasBeenSet; }
  template <typename DataTypeT = Aws::String>
  void SetDataType(DataTypeT&& value) {
    m_dataTypeHasBeenSet = true;
    m_dataType = std::forward<DataTypeT>(value);
  }
  template <typename DataTypeT = Aws::String>
  ParameterHistory& WithDataType(DataTypeT&& value) {
    SetDataType(std::forward<DataTypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  ParameterType m_type{ParameterType::NOT_SET};

  Aws::String m_keyId;

  Aws::Utils::DateTime m_lastModifiedDate{};

  Aws::String m_lastModifiedUser;

  Aws::String m_description;

  Aws::String m_value;

  Aws::String m_allowedPattern;

  long long m_version{0};

  Aws::Vector<Aws::String> m_labels;

  ParameterTier m_tier{ParameterTier::NOT_SET};

  Aws::Vector<ParameterInlinePolicy> m_policies;

  Aws::String m_dataType;
  bool m_nameHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_keyIdHasBeenSet = false;
  bool m_lastModifiedDateHasBeenSet = false;
  bool m_lastModifiedUserHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_valueHasBeenSet = false;
  bool m_allowedPatternHasBeenSet = false;
  bool m_versionHasBeenSet = false;
  bool m_labelsHasBeenSet = false;
  bool m_tierHasBeenSet = false;
  bool m_policiesHasBeenSet = false;
  bool m_dataTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
