# (c) Copyright 2009-2013, 2015. CodeWeavers, Inc.

import re
import os
import subprocess
import traceback

from gi.repository import GLib
from gi.repository import Gdk
from gi.repository import Gtk

import cxutils
import bottlequery
import cxobservable

import bottlecollection
import createshortcutdlg
import cxguilog
import cxguitools
import pyop

# for localization
from cxutils import cxgettext as _

# These are the observable events
DIALOG_CLOSED = 'dialog_closed'


#
# Dialog instance
#

class RunCommandController(cxobservable.Object):

    observable_events = frozenset((DIALOG_CLOSED,))

    def __init__(self, inBottleName):
        cxobservable.Object.__init__(self)

        #  Setup the GUI
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("cxrun"))
        self.xml.connect_signals(self)


        # Add indents to debug channel menu items
        for item in self.xml.get_object("DebugChannelMenu").get_children()[3:]:
            item.set_label('  '+item.get_label())


        #----------------------------------------------------------------------------
        #  Set up progress bar
        #----------------------------------------------------------------------------
        progbar = self.xml.get_object("ProgBar")
        progbar.set_pulse_step(.05)
        progbar.hide()


        #----------------------------------------------------------------------------
        #  Fill Bottle popup list
        #----------------------------------------------------------------------------
        bottlePopupWidget = self.xml.get_object("BottlePopup")
        self.bottleListStore = Gtk.ListStore(str, object) #display name, internal name
        bottlePopupWidget.set_model(self.bottleListStore)
        selectIter = 0
        collection = bottlecollection.sharedCollection()
        if collection.bottleList():
            for bottleName in sorted(collection.bottleList()):
                newrow = self.bottleListStore.append()
                self.bottleListStore.set_value(newrow, 0, bottleName)
                self.bottleListStore.set_value(newrow, 1, bottleName)
                if bottleName == inBottleName:
                    selectIter = newrow
        else:
            newrow = self.bottleListStore.append()
            self.bottleListStore.set_value(newrow, 0, _("(No bottles)"))
            self.bottleListStore.set_value(newrow, 1, None)

        if selectIter:
            bottlePopupWidget.set_active_iter(selectIter)
        else:
            bottlePopupWidget.set_active_iter(self.bottleListStore.get_iter_first())

        self.add_menu_open = False
        self.commandOp = None
        self.animateEvent = None
        self.set_widget_sensitivities(self)

    def set_command(self, command):
        commandEntry = self.xml.get_object("CommandEntry")
        commandEntry.set_text(command)

    def cancel_clicked(self, caller):
        self.xml.get_object("RunCommandDialog").destroy()
        self.quit_requested(caller)

    def quit_requested(self, _caller):
        self.emit_event(DIALOG_CLOSED)
        cxguitools.toplevel_quit()

    def progbar_pulse(self):
        progBar = self.xml.get_object("ProgBar")
        progBar.pulse()
        return True

    def set_widget_sensitivities(self, _caller):
        progBar = self.xml.get_object("ProgBar")

        collection = bottlecollection.sharedCollection()
        if self.commandOp is not None or not collection.bottleList():
            self.xml.get_object("RunCommandButton").set_sensitive(False)
            self.xml.get_object("CommandEntry").set_sensitive(False)
            self.xml.get_object("BottlePopup").set_sensitive(False)
            self.xml.get_object("DebugOptionsFrame").set_sensitive(False)
            self.xml.get_object("CommandBrowseButton").set_sensitive(False)
            progBar.show()
        else:
            if self.xml.get_object("CommandEntry").get_text():
                self.xml.get_object("RunCommandButton").set_sensitive(True)
            else:
                self.xml.get_object("RunCommandButton").set_sensitive(False)

            self.xml.get_object("CommandEntry").set_sensitive(True)
            self.xml.get_object("BottlePopup").set_sensitive(True)
            self.xml.get_object("DebugOptionsFrame").set_sensitive(True)
            self.xml.get_object("CommandBrowseButton").set_sensitive(True)
            progBar.hide()

        if self.xml.get_object("CommandEntry").get_text():
            self.xml.get_object("CreateIconButton").set_sensitive(True)
        else:
            self.xml.get_object("CreateIconButton").set_sensitive(False)

        if not collection.bottleList():
            self.xml.get_object("ProgBar").hide()

    def present(self):
        self.xml.get_object("RunCommandDialog").present()


    def run_command(self, _caller):
        logFileName = ""
        loggingChannels = ""
        env_string = ""
        if self.xml.get_object("LogFileCheckbox").get_active():
            fileSaver = Gtk.FileChooserDialog(
                title=_("Specify where to save the log file"),
                transient_for=self.xml.get_object("RunCommandDialog"),
                action=Gtk.FileChooserAction.SAVE)
            fileFilter = Gtk.FileFilter()
            fileFilter.add_pattern("*.cxlog")
            fileFilter.set_name(_("CrossOver Log Files"))
            fileSaver.add_filter(fileFilter)

            fileSaver.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
            fileSaver.add_button(Gtk.STOCK_SAVE, Gtk.ResponseType.OK)
            fileSaver.set_do_overwrite_confirmation(True)

            cxguitools.set_default_extension(fileSaver, 'cxlog')

            if fileSaver.run() == Gtk.ResponseType.OK:
                logFileName = fileSaver.get_filename()
                fileSaver.destroy()
            else:
                fileSaver.destroy()
                return

            loggingChannels = self.xml.get_object("DbgChannelEntry").get_text()

            env_string = self.xml.get_object("EnvStringEntry").get_text()

        self.animateEvent = GLib.timeout_add(100, self.progbar_pulse)

        command_line = self.xml.get_object("CommandEntry").get_text()

        if logFileName:
            log_file = cxguilog.create_log_file(logFileName, "Running command: %s\nBottle: %s\nDebug channels: %s\nExtra environment variables: %s\n\n" % (command_line, self.get_selected_bottle().name, loggingChannels, env_string))
        else:
            log_file = None

        self.commandOp = RunWineCommandOperation(self, command_line, self.get_selected_bottle().name, log_file, loggingChannels, env_string)
        self.set_widget_sensitivities(self)
        pyop.sharedOperationQueue.enqueue(self.commandOp)


    def toggle_logfile(self, widget):
        self.xml.get_object("ExtraLoggingChannelsFrame").set_sensitive(
            widget.get_active())


    def run_command_finished(self, _runCommandOp):
        self.commandOp = None
        self.set_widget_sensitivities(self)

        GLib.source_remove(self.animateEvent)


    def get_selected_bottle(self):
        collection = bottlecollection.sharedCollection()
        bottlePopup = self.xml.get_object("BottlePopup")
        activeIndex = bottlePopup.get_active_iter()
        selectedBottleName = self.bottleListStore.get_value(activeIndex, 1)
        if selectedBottleName:
            bottleObj = collection.bottleObject(selectedBottleName)
            return bottleObj
        return None


    def browse_for_command(self, _caller):
        filePicker = Gtk.FileChooserDialog(
            title=_("Choose a File to Run"),
            transient_for=self.xml.get_object("RunCommandDialog"),
            action=Gtk.FileChooserAction.OPEN)
        cxguitools.add_filters(filePicker, cxguitools.FILTERS_RUNNABLE | cxguitools.FILTERS_ALLFILES)

        selectedBottle = self.get_selected_bottle()
        environ = bottlequery.get_win_environ(selectedBottle.name, ("SystemDrive",))
        drive = bottlequery.expand_win_string(environ, "%SystemDrive%")
        drive = bottlequery.get_native_path(selectedBottle.name, drive)
        filePicker.set_current_folder(drive)

        filePicker.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
        filePicker.add_button(Gtk.STOCK_OPEN, Gtk.ResponseType.OK)

        if filePicker.run() == Gtk.ResponseType.OK:
            commandEntry = self.xml.get_object("CommandEntry")
            filename = cxutils.argvtocmdline((filePicker.get_filename(),))
            commandEntry.set_text(filename)

        filePicker.destroy()

    def create_icon(self, _widget):
        bottlename = self.get_selected_bottle().name
        command = self.xml.get_object("CommandEntry").get_text()
        createshortcutdlg.CreateShortcutDialog(bottlename=bottlename, command=command)

    def on_DbgChannelEntry_insert(self, widget, text, _length, _position):
        filtered_text = re.sub(r'[^\w\+\-,]', '', text)
        if text != filtered_text:
            position = widget.get_position()
            widget.insert_text(filtered_text, position)
            widget.stop_emission_by_name('insert_text')

    def on_DbgChannelButton_button_press_event(self, widget, event):
        if self.add_menu_open or event.button != 1:
            return False # propagate

        menu = self.xml.get_object("DebugChannelMenu")
        cxguitools.popup_at_widget(
            menu,
            self.xml.get_object("DbgChannelButton"),
            Gdk.Gravity.SOUTH_WEST,
            Gdk.Gravity.NORTH_WEST,
            event)

        self.add_menu_open = True

        widget.set_active(True)

        return True

    def on_DbgChannelButton_clicked(self, widget):
        if self.add_menu_open or not widget.get_active():
            return

        menu = self.xml.get_object("DebugChannelMenu")
        cxguitools.popup_at_widget(
            menu,
            self.xml.get_object("DbgChannelButton"),
            Gdk.Gravity.SOUTH_WEST,
            Gdk.Gravity.NORTH_WEST,
            None)

        self.add_menu_open = True

    def on_DebugChannelMenu_deactivate(self, _menushell):
        self.add_menu_open = False
        button = self.xml.get_object("DbgChannelButton")
        button.set_active(False)

    def on_DbgClearMenu_activate(self, _menu):
        self.xml.get_object("DbgChannelEntry").set_text('')
        self.xml.get_object("EnvStringEntry").set_text('')

    def add_debug_channels(self, menu):
        channels = self.xml.get_object("DbgChannelEntry").get_text().split(',')
        parts = menu.get_tooltip_text().split(' ')
        for channel in parts[0].split(','):
            if channel not in channels:
                channels.append(channel)
        if len(parts) > 1:
            self.xml.get_object("EnvStringEntry").set_text(' '.join(x for x in parts[1:] if x))
        self.xml.get_object("DbgChannelEntry").set_text(','.join(x for x in channels if x))

#
# Global dialog management
#

DIALOG = None

def _dialog_closed(_dialog, _event, _data):
    # pylint: disable=W0603
    global DIALOG
    DIALOG = None

def open_or_show(inBottleName):
    # pylint: disable=W0603
    global DIALOG
    if DIALOG is None:
        DIALOG = RunCommandController(inBottleName)
        DIALOG.add_observer(DIALOG_CLOSED, _dialog_closed)
    else:
        DIALOG.present()
    return DIALOG


#
# Operations
#

class RunWineCommandOperation(pyop.PythonOperation):

    def __init__(self, inCommandController, inCommand, inBottleName, inLogFile=None, inDebugChannels=None, inEnv=None):
        pyop.PythonOperation.__init__(self)
        self.bottlename = inBottleName
        self.commandController = inCommandController
        self.argArray = [os.path.join(cxutils.CX_ROOT, "bin", "cxstart")]
        self.argArray.extend(["--bottle", inBottleName, '--new-console'])
        self.log_file = inLogFile
        if inDebugChannels:
            self.argArray.append("--debugmsg")
            self.argArray.append(inDebugChannels)
        if inEnv:
            self.argArray.append("--env")
            self.argArray.append(inEnv)
        cmdline_array = cxutils.cmdlinetoargv(inCommand)
        dirname = cxutils.dirname(cmdline_array[0])
        if dirname:
            self.argArray.append('--workdir')
            self.argArray.append(dirname)
        self.argArray.append("--")
        self.argArray.extend(cxutils.cmdlinetoargv(inCommand))

    def __unicode__(self):
        return "RunWineCommandOperation for " + self.bottlename


    def main(self):
        try:
            if self.log_file:
                env = os.environ.copy()
                env['CX_LOG'] = '-'
                stderr = self.log_file
            else:
                env = None
                stderr = subprocess.PIPE
            subp = subprocess.Popen(self.argArray, stdout=subprocess.PIPE, stderr=stderr, env=env, universal_newlines=True, close_fds=True)
            if self.log_file:
                self.log_file.close()
            subp.communicate()
        except: # pylint: disable=W0702
            traceback.print_exc()


    def finish(self):
        self.commandController.run_command_finished(self)
        pyop.PythonOperation.finish(self)
